<?php
/**
 * JWT Authentication Handler
 */

class JWTHandler {
    private $secret_key;
    private $issuer;
    private $expiry_hours;

    public function __construct() {
        $this->secret_key = "ANILAD_COOP_SECRET_KEY_2024_SECURE";
        $this->issuer = "ANILAD_COOP";
        $this->expiry_hours = 24;
    }

    // Encode data to Base64URL
    private function base64UrlEncode($data) {
        return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
    }

    // Decode Base64URL
    private function base64UrlDecode($data) {
        return base64_decode(strtr($data, '-_', '+/'));
    }

    // Generate JWT Token
    public function generateToken($user_id, $email, $role = 'member') {
        $header = json_encode([
            'typ' => 'JWT',
            'alg' => 'HS256'
        ]);

        $payload = json_encode([
            'iss' => $this->issuer,
            'iat' => time(),
            'exp' => time() + ($this->expiry_hours * 3600),
            'user_id' => $user_id,
            'email' => $email,
            'role' => $role
        ]);

        $base64UrlHeader = $this->base64UrlEncode($header);
        $base64UrlPayload = $this->base64UrlEncode($payload);

        $signature = hash_hmac('sha256', $base64UrlHeader . "." . $base64UrlPayload, $this->secret_key, true);
        $base64UrlSignature = $this->base64UrlEncode($signature);

        return $base64UrlHeader . "." . $base64UrlPayload . "." . $base64UrlSignature;
    }

    // Validate JWT Token
    public function validateToken($token) {
        $tokenParts = explode('.', $token);
        
        if (count($tokenParts) !== 3) {
            return ['valid' => false, 'message' => 'Invalid token format'];
        }

        $header = $this->base64UrlDecode($tokenParts[0]);
        $payload = $this->base64UrlDecode($tokenParts[1]);
        $signatureProvided = $tokenParts[2];

        // Verify signature
        $signature = hash_hmac('sha256', $tokenParts[0] . "." . $tokenParts[1], $this->secret_key, true);
        $base64UrlSignature = $this->base64UrlEncode($signature);

        if ($base64UrlSignature !== $signatureProvided) {
            return ['valid' => false, 'message' => 'Invalid token signature'];
        }

        $payloadData = json_decode($payload, true);

        // Check expiry
        if (isset($payloadData['exp']) && time() > $payloadData['exp']) {
            return ['valid' => false, 'message' => 'Token has expired'];
        }

        return [
            'valid' => true,
            'data' => $payloadData
        ];
    }

    // Get token from headers
    public function getTokenFromHeaders() {
        $headers = getallheaders();
        
        if (isset($headers['Authorization'])) {
            $authHeader = $headers['Authorization'];
            if (preg_match('/Bearer\s(\S+)/', $authHeader, $matches)) {
                return $matches[1];
            }
        }
        
        return null;
    }

    // Middleware to check authentication
    public function authenticate() {
        $token = $this->getTokenFromHeaders();
        
        if (!$token) {
            http_response_code(401);
            echo json_encode(['success' => false, 'message' => 'No token provided']);
            exit;
        }

        $result = $this->validateToken($token);
        
        if (!$result['valid']) {
            http_response_code(401);
            echo json_encode(['success' => false, 'message' => $result['message']]);
            exit;
        }

        return $result['data'];
    }

    // Check if user is admin
    public function authenticateAdmin() {
        $userData = $this->authenticate();
        
        if ($userData['role'] !== 'admin' && $userData['role'] !== 'super_admin') {
            http_response_code(403);
            echo json_encode(['success' => false, 'message' => 'Admin access required']);
            exit;
        }

        return $userData;
    }
}
?>
