<?php
/**
 * Helper Functions
 */

// Generate unique member ID
function generateMemberId() {
    return 'MEM' . date('Y') . str_pad(rand(0, 9999), 4, '0', STR_PAD_LEFT);
}

// Generate virtual account number
function generateAccountNumber() {
    return '10' . str_pad(rand(0, 99999999), 8, '0', STR_PAD_LEFT);
}

// Generate loan reference
function generateLoanReference() {
    return 'LN' . date('Ymd') . str_pad(rand(0, 9999), 4, '0', STR_PAD_LEFT);
}

// Generate transaction reference
function generateTransactionReference() {
    return 'TXN' . date('YmdHis') . str_pad(rand(0, 999), 3, '0', STR_PAD_LEFT);
}

// Get interest rate based on repayment plan
function getInterestRate($months) {
    switch ($months) {
        case 3:
            return 3.0;
        case 6:
            return 5.0;
        case 12:
            return 10.0;
        default:
            return 0;
    }
}

// Calculate loan details
function calculateLoanDetails($amount, $months) {
    $interestRate = getInterestRate($months);
    $interest = ($amount * $interestRate) / 100;
    $totalAmount = $amount + $interest;
    $monthlyRepayment = $totalAmount / $months;
    
    return [
        'principal' => $amount,
        'interest_rate' => $interestRate,
        'interest_amount' => $interest,
        'total_amount' => $totalAmount,
        'monthly_repayment' => round($monthlyRepayment, 2)
    ];
}

// Check if member is eligible for loan (6 months membership)
function isMemberEligibleForLoan($memberCreatedAt) {
    $sixMonthsAgo = strtotime('-6 months');
    $memberDate = strtotime($memberCreatedAt);
    return $memberDate <= $sixMonthsAgo;
}

// Calculate maximum loan amount (200% of savings)
function calculateMaxLoanAmount($savingsBalance) {
    return $savingsBalance * 2;
}

// JSON response helper
function jsonResponse($data, $statusCode = 200) {
    http_response_code($statusCode);
    header('Content-Type: application/json');
    echo json_encode($data);
    exit;
}

// Validate email
function isValidEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

// Validate phone number (Nigerian format)
function isValidPhone($phone) {
    // Remove spaces and dashes
    $phone = preg_replace('/[\s-]/', '', $phone);
    // Check if it matches Nigerian phone format
    return preg_match('/^(\+234|234|0)[789][01]\d{8}$/', $phone);
}

// Sanitize input
function sanitizeInput($input) {
    return htmlspecialchars(strip_tags(trim($input)));
}

// Format currency
function formatCurrency($amount) {
    return '₦' . number_format($amount, 2);
}

// Get months since date
function getMonthsSince($date) {
    $datetime1 = new DateTime($date);
    $datetime2 = new DateTime();
    $interval = $datetime1->diff($datetime2);
    return ($interval->y * 12) + $interval->m;
}
?>
