<?php
/**
 * Get Member Transactions API
 * GET /api/transactions/list.php
 */

require_once '../../config/database.php';
require_once '../../includes/cors.php';
require_once '../../includes/helpers.php';
require_once '../../includes/jwt.php';

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    jsonResponse(['success' => false, 'message' => 'Method not allowed'], 405);
}

// Authenticate user
$jwt = new JWTHandler();
$userData = $jwt->authenticate();

// Get query parameters
$type = isset($_GET['type']) ? $_GET['type'] : null;
$limit = isset($_GET['limit']) ? intval($_GET['limit']) : 50;
$offset = isset($_GET['offset']) ? intval($_GET['offset']) : 0;

try {
    $database = new Database();
    $db = $database->getConnection();

    // Build query
    $whereClause = "WHERE t.member_id = ?";
    $params = [$userData['user_id']];

    if ($type && in_array($type, ['deposit', 'loan_repayment', 'loan_disbursement', 'withdrawal'])) {
        $whereClause .= " AND t.type = ?";
        $params[] = $type;
    }

    // Get total count
    $stmt = $db->prepare("SELECT COUNT(*) as total FROM transactions t $whereClause");
    $stmt->execute($params);
    $total = $stmt->fetch()['total'];

    // Get transactions
    $stmt = $db->prepare("
        SELECT t.*, l.loan_reference
        FROM transactions t
        LEFT JOIN loans l ON t.loan_id = l.id
        $whereClause
        ORDER BY t.created_at DESC
        LIMIT $limit OFFSET $offset
    ");
    $stmt->execute($params);
    $transactions = $stmt->fetchAll();

    $formattedTransactions = array_map(function($t) {
        return [
            'id' => $t['id'],
            'reference' => $t['transaction_reference'],
            'type' => $t['type'],
            'amount' => floatval($t['amount']),
            'balance_after' => floatval($t['balance_after']),
            'description' => $t['description'],
            'loan_reference' => $t['loan_reference'],
            'date' => $t['created_at']
        ];
    }, $transactions);

    jsonResponse([
        'success' => true,
        'data' => [
            'transactions' => $formattedTransactions,
            'total' => $total,
            'limit' => $limit,
            'offset' => $offset
        ]
    ]);

} catch (PDOException $e) {
    jsonResponse(['success' => false, 'message' => 'Failed to get transactions: ' . $e->getMessage()], 500);
}
?>
