<?php
/**
 * Member Registration API
 * POST /api/members/register.php
 */

require_once '../../config/database.php';
require_once '../../includes/cors.php';
require_once '../../includes/helpers.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonResponse(['success' => false, 'message' => 'Method not allowed'], 405);
}

// Get posted data
$data = json_decode(file_get_contents("php://input"), true);

// Validate required fields
$requiredFields = [
    'full_name' => 'Full name is required',
    'phone_number' => 'Phone number is required',
    'email' => 'Email address is required',
    'password' => 'Password is required',
    'state_of_origin' => 'State of origin is required',
    'next_of_kin_name' => 'Next of kin full name is required',
    'next_of_kin_phone' => 'Next of kin phone number is required'
];

foreach ($requiredFields as $field => $message) {
    if (empty($data[$field])) {
        jsonResponse(['success' => false, 'message' => $message], 400);
    }
}

// Sanitize inputs
$fullName = sanitizeInput($data['full_name']);
$phoneNumber = sanitizeInput($data['phone_number']);
$email = sanitizeInput($data['email']);
$password = $data['password'];
$stateOfOrigin = sanitizeInput($data['state_of_origin']);
$nextOfKinName = sanitizeInput($data['next_of_kin_name']);
$nextOfKinPhone = sanitizeInput($data['next_of_kin_phone']);
$nextOfKinEmail = isset($data['next_of_kin_email']) ? sanitizeInput($data['next_of_kin_email']) : null;

// Validate email
if (!isValidEmail($email)) {
    jsonResponse(['success' => false, 'message' => 'Invalid email address'], 400);
}

// Validate password length
if (strlen($password) < 6) {
    jsonResponse(['success' => false, 'message' => 'Password must be at least 6 characters'], 400);
}

// Validate next of kin email if provided
if ($nextOfKinEmail && !isValidEmail($nextOfKinEmail)) {
    jsonResponse(['success' => false, 'message' => 'Invalid next of kin email address'], 400);
}

try {
    $database = new Database();
    $db = $database->getConnection();

    // Check if email already exists
    $stmt = $db->prepare("SELECT id FROM members WHERE email = ?");
    $stmt->execute([$email]);
    if ($stmt->fetch()) {
        jsonResponse(['success' => false, 'message' => 'Email already registered'], 400);
    }

    // Check if phone already exists
    $stmt = $db->prepare("SELECT id FROM members WHERE phone_number = ?");
    $stmt->execute([$phoneNumber]);
    if ($stmt->fetch()) {
        jsonResponse(['success' => false, 'message' => 'Phone number already registered'], 400);
    }

    // Generate unique member ID and account number
    $memberId = generateMemberId();
    $accountNumber = generateAccountNumber();

    // Ensure member ID is unique
    $stmt = $db->prepare("SELECT id FROM members WHERE member_id = ?");
    $stmt->execute([$memberId]);
    while ($stmt->fetch()) {
        $memberId = generateMemberId();
        $stmt->execute([$memberId]);
    }

    // Ensure account number is unique
    $stmt = $db->prepare("SELECT id FROM virtual_accounts WHERE account_number = ?");
    $stmt->execute([$accountNumber]);
    while ($stmt->fetch()) {
        $accountNumber = generateAccountNumber();
        $stmt->execute([$accountNumber]);
    }

    // Hash password
    $hashedPassword = password_hash($password, PASSWORD_DEFAULT);

    // Begin transaction
    $db->beginTransaction();

    // Insert member
    $stmt = $db->prepare("
        INSERT INTO members (
            member_id, full_name, phone_number, email, password, 
            state_of_origin, next_of_kin_name, next_of_kin_phone, next_of_kin_email, status
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'pending')
    ");
    $stmt->execute([
        $memberId, $fullName, $phoneNumber, $email, $hashedPassword,
        $stateOfOrigin, $nextOfKinName, $nextOfKinPhone, $nextOfKinEmail
    ]);

    $memberDbId = $db->lastInsertId();

    // Create virtual account
    $stmt = $db->prepare("
        INSERT INTO virtual_accounts (member_id, account_number, account_name, savings_balance)
        VALUES (?, ?, ?, 0.00)
    ");
    $stmt->execute([$memberDbId, $accountNumber, $fullName]);

    // Commit transaction
    $db->commit();

    jsonResponse([
        'success' => true,
        'message' => 'Registration successful! Your account is pending approval.',
        'data' => [
            'member_id' => $memberId,
            'virtual_account' => [
                'account_number' => $accountNumber,
                'account_name' => $fullName,
                'bank_name' => 'ANILAD COOP Bank'
            ]
        ]
    ], 201);

} catch (PDOException $e) {
    if ($db->inTransaction()) {
        $db->rollBack();
    }
    jsonResponse(['success' => false, 'message' => 'Registration failed: ' . $e->getMessage()], 500);
}
?>
