<?php
/**
 * Get Member Profile API
 * GET /api/members/profile.php
 */

require_once '../../config/database.php';
require_once '../../includes/cors.php';
require_once '../../includes/helpers.php';
require_once '../../includes/jwt.php';

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    jsonResponse(['success' => false, 'message' => 'Method not allowed'], 405);
}

// Authenticate user
$jwt = new JWTHandler();
$userData = $jwt->authenticate();

try {
    $database = new Database();
    $db = $database->getConnection();

    // Get member with virtual account info
    $stmt = $db->prepare("
        SELECT m.*, va.account_number, va.savings_balance, va.bank_name
        FROM members m
        LEFT JOIN virtual_accounts va ON m.id = va.member_id
        WHERE m.id = ?
    ");
    $stmt->execute([$userData['user_id']]);
    $member = $stmt->fetch();

    if (!$member) {
        jsonResponse(['success' => false, 'message' => 'Member not found'], 404);
    }

    // Get loan eligibility
    $membershipMonths = getMonthsSince($member['created_at']);
    $isEligibleForLoan = $membershipMonths >= 6;
    $maxLoanAmount = calculateMaxLoanAmount($member['savings_balance']);

    // Get active loan
    $stmt = $db->prepare("
        SELECT * FROM loans 
        WHERE member_id = ? AND status IN ('pending', 'approved', 'disbursed', 'repaying')
        ORDER BY applied_at DESC LIMIT 1
    ");
    $stmt->execute([$member['id']]);
    $activeLoan = $stmt->fetch();

    // Get total loans taken
    $stmt = $db->prepare("
        SELECT COUNT(*) as total_loans, SUM(amount) as total_amount 
        FROM loans WHERE member_id = ? AND status = 'completed'
    ");
    $stmt->execute([$member['id']]);
    $loanStats = $stmt->fetch();

    jsonResponse([
        'success' => true,
        'data' => [
            'member' => [
                'id' => $member['id'],
                'member_id' => $member['member_id'],
                'full_name' => $member['full_name'],
                'email' => $member['email'],
                'phone_number' => $member['phone_number'],
                'state_of_origin' => $member['state_of_origin'],
                'next_of_kin_name' => $member['next_of_kin_name'],
                'next_of_kin_phone' => $member['next_of_kin_phone'],
                'next_of_kin_email' => $member['next_of_kin_email'],
                'status' => $member['status'],
                'member_since' => $member['created_at'],
                'membership_months' => $membershipMonths
            ],
            'virtual_account' => [
                'account_number' => $member['account_number'],
                'account_name' => $member['full_name'],
                'bank_name' => $member['bank_name'],
                'savings_balance' => floatval($member['savings_balance'])
            ],
            'loan_eligibility' => [
                'eligible' => $isEligibleForLoan,
                'months_remaining' => $isEligibleForLoan ? 0 : (6 - $membershipMonths),
                'max_loan_amount' => $maxLoanAmount,
                'has_active_loan' => $activeLoan !== false
            ],
            'active_loan' => $activeLoan ? [
                'loan_reference' => $activeLoan['loan_reference'],
                'amount' => floatval($activeLoan['amount']),
                'total_amount' => floatval($activeLoan['total_amount']),
                'amount_paid' => floatval($activeLoan['amount_paid']),
                'amount_remaining' => floatval($activeLoan['amount_remaining']),
                'monthly_repayment' => floatval($activeLoan['monthly_repayment']),
                'status' => $activeLoan['status']
            ] : null,
            'loan_stats' => [
                'total_loans_completed' => intval($loanStats['total_loans']),
                'total_amount_borrowed' => floatval($loanStats['total_amount'] ?? 0)
            ]
        ]
    ]);

} catch (PDOException $e) {
    jsonResponse(['success' => false, 'message' => 'Failed to get profile: ' . $e->getMessage()], 500);
}
?>
