<?php
/**
 * Member Login API
 * POST /api/members/login.php
 */

require_once '../../config/database.php';
require_once '../../includes/cors.php';
require_once '../../includes/helpers.php';
require_once '../../includes/jwt.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonResponse(['success' => false, 'message' => 'Method not allowed'], 405);
}

$data = json_decode(file_get_contents("php://input"), true);

if (empty($data['email']) || empty($data['password'])) {
    jsonResponse(['success' => false, 'message' => 'Email and password are required'], 400);
}

$email = sanitizeInput($data['email']);
$password = $data['password'];

try {
    $database = new Database();
    $db = $database->getConnection();

    // Get member with virtual account info
    $stmt = $db->prepare("
        SELECT m.*, va.account_number, va.savings_balance, va.bank_name
        FROM members m
        LEFT JOIN virtual_accounts va ON m.id = va.member_id
        WHERE m.email = ?
    ");
    $stmt->execute([$email]);
    $member = $stmt->fetch();

    if (!$member) {
        jsonResponse(['success' => false, 'message' => 'Invalid email or password'], 401);
    }

    if (!password_verify($password, $member['password'])) {
        jsonResponse(['success' => false, 'message' => 'Invalid email or password'], 401);
    }

    if ($member['status'] === 'pending') {
        jsonResponse(['success' => false, 'message' => 'Your account is pending approval'], 403);
    }

    if ($member['status'] === 'suspended') {
        jsonResponse(['success' => false, 'message' => 'Your account has been suspended'], 403);
    }

    if ($member['status'] === 'rejected') {
        jsonResponse(['success' => false, 'message' => 'Your registration was rejected'], 403);
    }

    // Generate JWT token
    $jwt = new JWTHandler();
    $token = $jwt->generateToken($member['id'], $member['email'], 'member');

    // Get active loan if any
    $stmt = $db->prepare("
        SELECT id, loan_reference, amount_remaining, status 
        FROM loans 
        WHERE member_id = ? AND status IN ('disbursed', 'repaying')
        ORDER BY created_at DESC LIMIT 1
    ");
    $stmt->execute([$member['id']]);
    $activeLoan = $stmt->fetch();

    jsonResponse([
        'success' => true,
        'message' => 'Login successful',
        'data' => [
            'token' => $token,
            'member' => [
                'id' => $member['id'],
                'member_id' => $member['member_id'],
                'full_name' => $member['full_name'],
                'email' => $member['email'],
                'phone_number' => $member['phone_number'],
                'state_of_origin' => $member['state_of_origin'],
                'status' => $member['status'],
                'member_since' => $member['created_at']
            ],
            'virtual_account' => [
                'account_number' => $member['account_number'],
                'account_name' => $member['full_name'],
                'bank_name' => $member['bank_name'],
                'savings_balance' => floatval($member['savings_balance'])
            ],
            'active_loan' => $activeLoan ? [
                'loan_reference' => $activeLoan['loan_reference'],
                'amount_remaining' => floatval($activeLoan['amount_remaining']),
                'status' => $activeLoan['status']
            ] : null
        ]
    ]);

} catch (PDOException $e) {
    jsonResponse(['success' => false, 'message' => 'Login failed: ' . $e->getMessage()], 500);
}
?>
