<?php
/**
 * Get Member Loans API
 * GET /api/loans/list.php
 */

require_once '../../config/database.php';
require_once '../../includes/cors.php';
require_once '../../includes/helpers.php';
require_once '../../includes/jwt.php';

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    jsonResponse(['success' => false, 'message' => 'Method not allowed'], 405);
}

// Authenticate user
$jwt = new JWTHandler();
$userData = $jwt->authenticate();

try {
    $database = new Database();
    $db = $database->getConnection();

    // Get all loans for the member
    $stmt = $db->prepare("
        SELECT l.*, 
               (SELECT GROUP_CONCAT(CONCAT(guarantor_name, ':', guarantor_phone) SEPARATOR '|') 
                FROM loan_guarantors WHERE loan_id = l.id) as guarantors
        FROM loans l
        WHERE l.member_id = ?
        ORDER BY l.applied_at DESC
    ");
    $stmt->execute([$userData['user_id']]);
    $loans = $stmt->fetchAll();

    $formattedLoans = [];
    foreach ($loans as $loan) {
        // Parse guarantors
        $guarantors = [];
        if ($loan['guarantors']) {
            $guarantorParts = explode('|', $loan['guarantors']);
            foreach ($guarantorParts as $part) {
                list($name, $phone) = explode(':', $part);
                $guarantors[] = ['name' => $name, 'phone' => $phone];
            }
        }

        // Get repayment history
        $stmt = $db->prepare("
            SELECT lr.*, t.transaction_reference, t.created_at as payment_date
            FROM loan_repayments lr
            JOIN transactions t ON lr.transaction_id = t.id
            WHERE lr.loan_id = ?
            ORDER BY lr.created_at DESC
        ");
        $stmt->execute([$loan['id']]);
        $repayments = $stmt->fetchAll();

        $formattedLoans[] = [
            'id' => $loan['id'],
            'loan_reference' => $loan['loan_reference'],
            'amount' => floatval($loan['amount']),
            'interest_rate' => floatval($loan['interest_rate']),
            'total_amount' => floatval($loan['total_amount']),
            'repayment_plan' => $loan['repayment_plan'] . ' months',
            'monthly_repayment' => floatval($loan['monthly_repayment']),
            'amount_paid' => floatval($loan['amount_paid']),
            'amount_remaining' => floatval($loan['amount_remaining']),
            'receiving_account' => [
                'account_number' => $loan['receiving_account_number'],
                'bank_name' => $loan['receiving_bank_name'],
                'account_name' => $loan['receiving_account_name']
            ],
            'status' => $loan['status'],
            'guarantors' => $guarantors,
            'applied_at' => $loan['applied_at'],
            'approved_at' => $loan['approved_at'],
            'disbursed_at' => $loan['disbursed_at'],
            'completed_at' => $loan['completed_at'],
            'repayment_history' => array_map(function($r) {
                return [
                    'reference' => $r['transaction_reference'],
                    'amount' => floatval($r['amount']),
                    'balance_after' => floatval($r['balance_after']),
                    'date' => $r['payment_date']
                ];
            }, $repayments)
        ];
    }

    jsonResponse([
        'success' => true,
        'data' => [
            'loans' => $formattedLoans,
            'total' => count($formattedLoans)
        ]
    ]);

} catch (PDOException $e) {
    jsonResponse(['success' => false, 'message' => 'Failed to get loans: ' . $e->getMessage()], 500);
}
?>
