<?php
/**
 * Loan Application API
 * POST /api/loans/apply.php
 */

require_once '../../config/database.php';
require_once '../../includes/cors.php';
require_once '../../includes/helpers.php';
require_once '../../includes/jwt.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonResponse(['success' => false, 'message' => 'Method not allowed'], 405);
}

// Authenticate user
$jwt = new JWTHandler();
$userData = $jwt->authenticate();

$data = json_decode(file_get_contents("php://input"), true);

// Validate required fields
$requiredFields = [
    'amount' => 'Loan amount is required',
    'repayment_plan' => 'Repayment plan is required',
    'guarantors' => 'Guarantors information is required',
    'receiving_account_number' => 'Receiving account number is required',
    'receiving_bank_name' => 'Receiving bank name is required',
    'receiving_account_name' => 'Receiving account name is required'
];

foreach ($requiredFields as $field => $message) {
    if (empty($data[$field])) {
        jsonResponse(['success' => false, 'message' => $message], 400);
    }
}

$amount = floatval($data['amount']);
$repaymentPlan = intval($data['repayment_plan']);
$guarantors = $data['guarantors'];
$receivingAccountNumber = sanitizeInput($data['receiving_account_number']);
$receivingBankName = sanitizeInput($data['receiving_bank_name']);
$receivingAccountName = sanitizeInput($data['receiving_account_name']);

// Validate repayment plan
if (!in_array($repaymentPlan, [3, 6, 12])) {
    jsonResponse(['success' => false, 'message' => 'Invalid repayment plan. Choose 3, 6, or 12 months'], 400);
}

// Validate guarantors (must be exactly 3)
if (!is_array($guarantors) || count($guarantors) !== 3) {
    jsonResponse(['success' => false, 'message' => 'Exactly 3 guarantors are required'], 400);
}

foreach ($guarantors as $index => $guarantor) {
    if (empty($guarantor['name']) || empty($guarantor['phone'])) {
        jsonResponse(['success' => false, 'message' => 'Guarantor ' . ($index + 1) . ' name and phone are required'], 400);
    }
}

try {
    $database = new Database();
    $db = $database->getConnection();

    // Get member info with savings balance
    $stmt = $db->prepare("
        SELECT m.*, va.savings_balance
        FROM members m
        JOIN virtual_accounts va ON m.id = va.member_id
        WHERE m.id = ?
    ");
    $stmt->execute([$userData['user_id']]);
    $member = $stmt->fetch();

    if (!$member) {
        jsonResponse(['success' => false, 'message' => 'Member not found'], 404);
    }

    if ($member['status'] !== 'approved') {
        jsonResponse(['success' => false, 'message' => 'Your account must be approved to apply for a loan'], 403);
    }

    // Check membership duration (minimum 6 months)
    if (!isMemberEligibleForLoan($member['created_at'])) {
        $monthsSince = getMonthsSince($member['created_at']);
        $monthsRemaining = 6 - $monthsSince;
        jsonResponse([
            'success' => false, 
            'message' => "You must be a member for at least 6 months. You have $monthsRemaining months remaining."
        ], 403);
    }

    // Check for existing active loan
    $stmt = $db->prepare("
        SELECT id FROM loans 
        WHERE member_id = ? AND status IN ('pending', 'approved', 'disbursed', 'repaying')
    ");
    $stmt->execute([$member['id']]);
    if ($stmt->fetch()) {
        jsonResponse(['success' => false, 'message' => 'You already have an active loan'], 403);
    }

    // Check maximum loan amount (200% of savings)
    $maxLoanAmount = calculateMaxLoanAmount($member['savings_balance']);
    if ($amount > $maxLoanAmount) {
        jsonResponse([
            'success' => false, 
            'message' => "Maximum loan amount is ₦" . number_format($maxLoanAmount, 2) . " (200% of your savings)"
        ], 400);
    }

    if ($amount <= 0) {
        jsonResponse(['success' => false, 'message' => 'Invalid loan amount'], 400);
    }

    // Calculate loan details
    $loanDetails = calculateLoanDetails($amount, $repaymentPlan);

    // Generate unique loan reference
    $loanReference = generateLoanReference();
    $stmt = $db->prepare("SELECT id FROM loans WHERE loan_reference = ?");
    $stmt->execute([$loanReference]);
    while ($stmt->fetch()) {
        $loanReference = generateLoanReference();
        $stmt->execute([$loanReference]);
    }

    // Begin transaction
    $db->beginTransaction();

    // Insert loan
    $stmt = $db->prepare("
        INSERT INTO loans (
            loan_reference, member_id, amount, interest_rate, total_amount,
            repayment_plan, monthly_repayment, amount_remaining,
            receiving_account_number, receiving_bank_name, receiving_account_name, status
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'pending')
    ");
    $stmt->execute([
        $loanReference,
        $member['id'],
        $amount,
        $loanDetails['interest_rate'],
        $loanDetails['total_amount'],
        $repaymentPlan,
        $loanDetails['monthly_repayment'],
        $loanDetails['total_amount'],
        $receivingAccountNumber,
        $receivingBankName,
        $receivingAccountName
    ]);

    $loanId = $db->lastInsertId();

    // Insert guarantors
    $stmt = $db->prepare("
        INSERT INTO loan_guarantors (loan_id, guarantor_name, guarantor_phone)
        VALUES (?, ?, ?)
    ");
    
    foreach ($guarantors as $guarantor) {
        $stmt->execute([
            $loanId,
            sanitizeInput($guarantor['name']),
            sanitizeInput($guarantor['phone'])
        ]);
    }

    // Commit transaction
    $db->commit();

    jsonResponse([
        'success' => true,
        'message' => 'Loan application submitted successfully. Awaiting admin approval.',
        'data' => [
            'loan_reference' => $loanReference,
            'principal_amount' => $amount,
            'interest_rate' => $loanDetails['interest_rate'],
            'interest_amount' => $loanDetails['interest_amount'],
            'total_amount' => $loanDetails['total_amount'],
            'repayment_plan' => $repaymentPlan . ' months',
            'monthly_repayment' => $loanDetails['monthly_repayment']
        ]
    ], 201);

} catch (PDOException $e) {
    if ($db->inTransaction()) {
        $db->rollBack();
    }
    jsonResponse(['success' => false, 'message' => 'Loan application failed: ' . $e->getMessage()], 500);
}
?>
