<?php
/**
 * Process Deposit API
 * POST /api/deposits/process.php
 * 
 * This handles deposits to virtual accounts.
 * Logic: Check for active loan first, repay loan, then add remainder to savings.
 */

require_once '../../config/database.php';
require_once '../../includes/cors.php';
require_once '../../includes/helpers.php';
require_once '../../includes/jwt.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonResponse(['success' => false, 'message' => 'Method not allowed'], 405);
}

$data = json_decode(file_get_contents("php://input"), true);

// Validate required fields
if (empty($data['account_number']) || empty($data['amount'])) {
    jsonResponse(['success' => false, 'message' => 'Account number and amount are required'], 400);
}

$accountNumber = sanitizeInput($data['account_number']);
$amount = floatval($data['amount']);

if ($amount <= 0) {
    jsonResponse(['success' => false, 'message' => 'Invalid amount'], 400);
}

try {
    $database = new Database();
    $db = $database->getConnection();

    // Get virtual account and member
    $stmt = $db->prepare("
        SELECT va.*, m.id as member_id, m.full_name, m.status as member_status
        FROM virtual_accounts va
        JOIN members m ON va.member_id = m.id
        WHERE va.account_number = ?
    ");
    $stmt->execute([$accountNumber]);
    $account = $stmt->fetch();

    if (!$account) {
        jsonResponse(['success' => false, 'message' => 'Account not found'], 404);
    }

    if ($account['member_status'] !== 'approved') {
        jsonResponse(['success' => false, 'message' => 'Member account is not approved'], 403);
    }

    // Begin transaction
    $db->beginTransaction();

    $remainingAmount = $amount;
    $loanRepaid = 0;
    $addedToSavings = 0;
    $transactions = [];

    // Check for active loan (disbursed or repaying)
    $stmt = $db->prepare("
        SELECT * FROM loans 
        WHERE member_id = ? AND status IN ('disbursed', 'repaying')
        ORDER BY disbursed_at ASC LIMIT 1
    ");
    $stmt->execute([$account['member_id']]);
    $activeLoan = $stmt->fetch();

    if ($activeLoan && $activeLoan['amount_remaining'] > 0) {
        // Calculate repayment amount
        $repaymentAmount = min($remainingAmount, $activeLoan['amount_remaining']);
        $remainingAmount -= $repaymentAmount;
        $loanRepaid = $repaymentAmount;

        // Generate transaction reference
        $txnReference = generateTransactionReference();

        // Update loan
        $newAmountPaid = $activeLoan['amount_paid'] + $repaymentAmount;
        $newAmountRemaining = $activeLoan['amount_remaining'] - $repaymentAmount;
        $newLoanStatus = $newAmountRemaining <= 0 ? 'completed' : 'repaying';
        $completedAt = $newLoanStatus === 'completed' ? date('Y-m-d H:i:s') : null;

        $stmt = $db->prepare("
            UPDATE loans 
            SET amount_paid = ?, amount_remaining = ?, status = ?, completed_at = ?
            WHERE id = ?
        ");
        $stmt->execute([$newAmountPaid, $newAmountRemaining, $newLoanStatus, $completedAt, $activeLoan['id']]);

        // Record loan repayment transaction
        $stmt = $db->prepare("
            INSERT INTO transactions (
                transaction_reference, member_id, virtual_account_id, 
                type, amount, balance_after, description, loan_id
            ) VALUES (?, ?, ?, 'loan_repayment', ?, ?, ?, ?)
        ");
        $stmt->execute([
            $txnReference,
            $account['member_id'],
            $account['id'],
            $repaymentAmount,
            $account['savings_balance'], // Balance unchanged for loan repayment
            "Loan repayment for " . $activeLoan['loan_reference'],
            $activeLoan['id']
        ]);
        $transactionId = $db->lastInsertId();

        // Record in loan_repayments
        $stmt = $db->prepare("
            INSERT INTO loan_repayments (
                loan_id, transaction_id, amount, principal_paid, interest_paid, balance_after
            ) VALUES (?, ?, ?, ?, ?, ?)
        ");
        
        // Calculate principal and interest portion (simplified - proportional)
        $totalPrincipal = $activeLoan['amount'];
        $totalInterest = $activeLoan['total_amount'] - $activeLoan['amount'];
        $ratio = $repaymentAmount / $activeLoan['total_amount'];
        $principalPaid = $totalPrincipal * $ratio;
        $interestPaid = $totalInterest * $ratio;
        
        $stmt->execute([
            $activeLoan['id'],
            $transactionId,
            $repaymentAmount,
            $principalPaid,
            $interestPaid,
            $newAmountRemaining
        ]);

        $transactions[] = [
            'type' => 'loan_repayment',
            'reference' => $txnReference,
            'amount' => $repaymentAmount,
            'loan_reference' => $activeLoan['loan_reference'],
            'loan_balance' => $newAmountRemaining
        ];
    }

    // Add remaining amount to savings
    if ($remainingAmount > 0) {
        $addedToSavings = $remainingAmount;
        $newSavingsBalance = $account['savings_balance'] + $remainingAmount;

        // Update savings balance
        $stmt = $db->prepare("
            UPDATE virtual_accounts SET savings_balance = ? WHERE id = ?
        ");
        $stmt->execute([$newSavingsBalance, $account['id']]);

        // Generate transaction reference
        $txnReference = generateTransactionReference();

        // Record deposit transaction
        $stmt = $db->prepare("
            INSERT INTO transactions (
                transaction_reference, member_id, virtual_account_id, 
                type, amount, balance_after, description
            ) VALUES (?, ?, ?, 'deposit', ?, ?, ?)
        ");
        $stmt->execute([
            $txnReference,
            $account['member_id'],
            $account['id'],
            $remainingAmount,
            $newSavingsBalance,
            "Deposit to savings"
        ]);

        $transactions[] = [
            'type' => 'deposit',
            'reference' => $txnReference,
            'amount' => $remainingAmount,
            'new_balance' => $newSavingsBalance
        ];
    }

    // Commit transaction
    $db->commit();

    jsonResponse([
        'success' => true,
        'message' => 'Deposit processed successfully',
        'data' => [
            'total_deposited' => $amount,
            'loan_repaid' => $loanRepaid,
            'added_to_savings' => $addedToSavings,
            'transactions' => $transactions
        ]
    ]);

} catch (PDOException $e) {
    if ($db->inTransaction()) {
        $db->rollBack();
    }
    jsonResponse(['success' => false, 'message' => 'Deposit failed: ' . $e->getMessage()], 500);
}
?>
