<?php
/**
 * Admin - List All Transactions API
 * GET /api/admin/transactions/list.php
 */

require_once '../../../config/database.php';
require_once '../../../includes/cors.php';
require_once '../../../includes/helpers.php';
require_once '../../../includes/jwt.php';

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    jsonResponse(['success' => false, 'message' => 'Method not allowed'], 405);
}

// Authenticate admin
$jwt = new JWTHandler();
$userData = $jwt->authenticateAdmin();

// Get query parameters
$type = isset($_GET['type']) ? $_GET['type'] : null;
$search = isset($_GET['search']) ? $_GET['search'] : null;
$startDate = isset($_GET['start_date']) ? $_GET['start_date'] : null;
$endDate = isset($_GET['end_date']) ? $_GET['end_date'] : null;
$limit = isset($_GET['limit']) ? intval($_GET['limit']) : 50;
$offset = isset($_GET['offset']) ? intval($_GET['offset']) : 0;

try {
    $database = new Database();
    $db = $database->getConnection();

    // Build query
    $whereClause = "WHERE 1=1";
    $params = [];

    if ($type && in_array($type, ['deposit', 'loan_repayment', 'loan_disbursement', 'withdrawal'])) {
        $whereClause .= " AND t.type = ?";
        $params[] = $type;
    }

    if ($search) {
        $whereClause .= " AND (t.transaction_reference LIKE ? OR m.full_name LIKE ? OR m.member_id LIKE ?)";
        $searchParam = "%$search%";
        $params = array_merge($params, [$searchParam, $searchParam, $searchParam]);
    }

    if ($startDate) {
        $whereClause .= " AND DATE(t.created_at) >= ?";
        $params[] = $startDate;
    }

    if ($endDate) {
        $whereClause .= " AND DATE(t.created_at) <= ?";
        $params[] = $endDate;
    }

    // Get total count
    $stmt = $db->prepare("
        SELECT COUNT(*) as total 
        FROM transactions t
        JOIN members m ON t.member_id = m.id
        $whereClause
    ");
    $stmt->execute($params);
    $total = $stmt->fetch()['total'];

    // Get transactions
    $stmt = $db->prepare("
        SELECT t.*, m.member_id as member_code, m.full_name, l.loan_reference
        FROM transactions t
        JOIN members m ON t.member_id = m.id
        LEFT JOIN loans l ON t.loan_id = l.id
        $whereClause
        ORDER BY t.created_at DESC
        LIMIT $limit OFFSET $offset
    ");
    $stmt->execute($params);
    $transactions = $stmt->fetchAll();

    $formattedTransactions = array_map(function($t) {
        return [
            'id' => $t['id'],
            'reference' => $t['transaction_reference'],
            'member' => [
                'member_id' => $t['member_code'],
                'full_name' => $t['full_name']
            ],
            'type' => $t['type'],
            'amount' => floatval($t['amount']),
            'balance_after' => floatval($t['balance_after']),
            'description' => $t['description'],
            'loan_reference' => $t['loan_reference'],
            'date' => $t['created_at']
        ];
    }, $transactions);

    // Get summary
    $stmt = $db->prepare("
        SELECT type, COUNT(*) as count, SUM(amount) as total
        FROM transactions t
        JOIN members m ON t.member_id = m.id
        $whereClause
        GROUP BY type
    ");
    $stmt->execute($params);
    $summary = $stmt->fetchAll();

    jsonResponse([
        'success' => true,
        'data' => [
            'transactions' => $formattedTransactions,
            'summary' => $summary,
            'total' => $total,
            'limit' => $limit,
            'offset' => $offset
        ]
    ]);

} catch (PDOException $e) {
    jsonResponse(['success' => false, 'message' => 'Failed to get transactions: ' . $e->getMessage()], 500);
}
?>
