<?php
/**
 * Admin - List All Members API
 * GET /api/admin/members/list.php
 */

require_once '../../../config/database.php';
require_once '../../../includes/cors.php';
require_once '../../../includes/helpers.php';
require_once '../../../includes/jwt.php';

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    jsonResponse(['success' => false, 'message' => 'Method not allowed'], 405);
}

// Authenticate admin
$jwt = new JWTHandler();
$userData = $jwt->authenticateAdmin();

// Get query parameters
$status = isset($_GET['status']) ? $_GET['status'] : null;
$search = isset($_GET['search']) ? $_GET['search'] : null;
$limit = isset($_GET['limit']) ? intval($_GET['limit']) : 50;
$offset = isset($_GET['offset']) ? intval($_GET['offset']) : 0;

try {
    $database = new Database();
    $db = $database->getConnection();

    // Build query
    $whereClause = "WHERE 1=1";
    $params = [];

    if ($status && in_array($status, ['pending', 'approved', 'suspended', 'rejected'])) {
        $whereClause .= " AND m.status = ?";
        $params[] = $status;
    }

    if ($search) {
        $whereClause .= " AND (m.full_name LIKE ? OR m.email LIKE ? OR m.member_id LIKE ? OR m.phone_number LIKE ?)";
        $searchParam = "%$search%";
        $params = array_merge($params, [$searchParam, $searchParam, $searchParam, $searchParam]);
    }

    // Get total count
    $stmt = $db->prepare("SELECT COUNT(*) as total FROM members m $whereClause");
    $stmt->execute($params);
    $total = $stmt->fetch()['total'];

    // Get members with virtual account info
    $stmt = $db->prepare("
        SELECT m.*, va.account_number, va.savings_balance,
               a.full_name as approved_by_name
        FROM members m
        LEFT JOIN virtual_accounts va ON m.id = va.member_id
        LEFT JOIN admins a ON m.approved_by = a.id
        $whereClause
        ORDER BY m.created_at DESC
        LIMIT $limit OFFSET $offset
    ");
    $stmt->execute($params);
    $members = $stmt->fetchAll();

    $formattedMembers = array_map(function($m) {
        return [
            'id' => $m['id'],
            'member_id' => $m['member_id'],
            'full_name' => $m['full_name'],
            'email' => $m['email'],
            'phone_number' => $m['phone_number'],
            'state_of_origin' => $m['state_of_origin'],
            'next_of_kin' => [
                'name' => $m['next_of_kin_name'],
                'phone' => $m['next_of_kin_phone'],
                'email' => $m['next_of_kin_email']
            ],
            'virtual_account' => [
                'account_number' => $m['account_number'],
                'savings_balance' => floatval($m['savings_balance'])
            ],
            'status' => $m['status'],
            'approved_by' => $m['approved_by_name'],
            'approved_at' => $m['approved_at'],
            'created_at' => $m['created_at'],
            'membership_months' => getMonthsSince($m['created_at'])
        ];
    }, $members);

    jsonResponse([
        'success' => true,
        'data' => [
            'members' => $formattedMembers,
            'total' => $total,
            'limit' => $limit,
            'offset' => $offset
        ]
    ]);

} catch (PDOException $e) {
    jsonResponse(['success' => false, 'message' => 'Failed to get members: ' . $e->getMessage()], 500);
}
?>
