<?php
/**
 * Admin - List All Loans API
 * GET /api/admin/loans/list.php
 */

require_once '../../../config/database.php';
require_once '../../../includes/cors.php';
require_once '../../../includes/helpers.php';
require_once '../../../includes/jwt.php';

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    jsonResponse(['success' => false, 'message' => 'Method not allowed'], 405);
}

// Authenticate admin
$jwt = new JWTHandler();
$userData = $jwt->authenticateAdmin();

// Get query parameters
$status = isset($_GET['status']) ? $_GET['status'] : null;
$search = isset($_GET['search']) ? $_GET['search'] : null;
$limit = isset($_GET['limit']) ? intval($_GET['limit']) : 50;
$offset = isset($_GET['offset']) ? intval($_GET['offset']) : 0;

try {
    $database = new Database();
    $db = $database->getConnection();

    // Build query
    $whereClause = "WHERE 1=1";
    $params = [];

    if ($status && in_array($status, ['pending', 'approved', 'disbursed', 'repaying', 'completed', 'rejected', 'defaulted'])) {
        $whereClause .= " AND l.status = ?";
        $params[] = $status;
    }

    if ($search) {
        $whereClause .= " AND (l.loan_reference LIKE ? OR m.full_name LIKE ? OR m.member_id LIKE ?)";
        $searchParam = "%$search%";
        $params = array_merge($params, [$searchParam, $searchParam, $searchParam]);
    }

    // Get total count
    $stmt = $db->prepare("
        SELECT COUNT(*) as total 
        FROM loans l 
        JOIN members m ON l.member_id = m.id 
        $whereClause
    ");
    $stmt->execute($params);
    $total = $stmt->fetch()['total'];

    // Get loans with member info
    $stmt = $db->prepare("
        SELECT l.*, m.member_id as member_code, m.full_name, m.email, m.phone_number,
               va.savings_balance,
               a.full_name as approved_by_name,
               (SELECT GROUP_CONCAT(CONCAT(guarantor_name, ':', guarantor_phone) SEPARATOR '|') 
                FROM loan_guarantors WHERE loan_id = l.id) as guarantors
        FROM loans l
        JOIN members m ON l.member_id = m.id
        LEFT JOIN virtual_accounts va ON m.id = va.member_id
        LEFT JOIN admins a ON l.approved_by = a.id
        $whereClause
        ORDER BY l.applied_at DESC
        LIMIT $limit OFFSET $offset
    ");
    $stmt->execute($params);
    $loans = $stmt->fetchAll();

    $formattedLoans = array_map(function($loan) {
        // Parse guarantors
        $guarantors = [];
        if ($loan['guarantors']) {
            $guarantorParts = explode('|', $loan['guarantors']);
            foreach ($guarantorParts as $part) {
                list($name, $phone) = explode(':', $part);
                $guarantors[] = ['name' => $name, 'phone' => $phone];
            }
        }

        return [
            'id' => $loan['id'],
            'loan_reference' => $loan['loan_reference'],
            'member' => [
                'id' => $loan['member_id'],
                'member_id' => $loan['member_code'],
                'full_name' => $loan['full_name'],
                'email' => $loan['email'],
                'phone_number' => $loan['phone_number'],
                'savings_balance' => floatval($loan['savings_balance'])
            ],
            'amount' => floatval($loan['amount']),
            'interest_rate' => floatval($loan['interest_rate']),
            'total_amount' => floatval($loan['total_amount']),
            'repayment_plan' => $loan['repayment_plan'] . ' months',
            'monthly_repayment' => floatval($loan['monthly_repayment']),
            'amount_paid' => floatval($loan['amount_paid']),
            'amount_remaining' => floatval($loan['amount_remaining']),
            'receiving_account' => [
                'account_number' => $loan['receiving_account_number'],
                'bank_name' => $loan['receiving_bank_name'],
                'account_name' => $loan['receiving_account_name']
            ],
            'guarantors' => $guarantors,
            'status' => $loan['status'],
            'approved_by' => $loan['approved_by_name'],
            'applied_at' => $loan['applied_at'],
            'approved_at' => $loan['approved_at'],
            'disbursed_at' => $loan['disbursed_at'],
            'completed_at' => $loan['completed_at']
        ];
    }, $loans);

    jsonResponse([
        'success' => true,
        'data' => [
            'loans' => $formattedLoans,
            'total' => $total,
            'limit' => $limit,
            'offset' => $offset
        ]
    ]);

} catch (PDOException $e) {
    jsonResponse(['success' => false, 'message' => 'Failed to get loans: ' . $e->getMessage()], 500);
}
?>
