<?php
/**
 * Admin - Disburse Loan API
 * PUT /api/admin/loans/disburse.php
 */

require_once '../../../config/database.php';
require_once '../../../includes/cors.php';
require_once '../../../includes/helpers.php';
require_once '../../../includes/jwt.php';

if ($_SERVER['REQUEST_METHOD'] !== 'PUT') {
    jsonResponse(['success' => false, 'message' => 'Method not allowed'], 405);
}

// Authenticate admin
$jwt = new JWTHandler();
$userData = $jwt->authenticateAdmin();

$data = json_decode(file_get_contents("php://input"), true);

if (empty($data['loan_id'])) {
    jsonResponse(['success' => false, 'message' => 'Loan ID is required'], 400);
}

$loanId = intval($data['loan_id']);

try {
    $database = new Database();
    $db = $database->getConnection();

    // Check if loan exists and is approved
    $stmt = $db->prepare("
        SELECT l.*, m.id as member_db_id, m.full_name, m.member_id as member_code, va.id as va_id
        FROM loans l
        JOIN members m ON l.member_id = m.id
        JOIN virtual_accounts va ON m.id = va.member_id
        WHERE l.id = ?
    ");
    $stmt->execute([$loanId]);
    $loan = $stmt->fetch();

    if (!$loan) {
        jsonResponse(['success' => false, 'message' => 'Loan not found'], 404);
    }

    if ($loan['status'] !== 'approved') {
        jsonResponse(['success' => false, 'message' => 'Only approved loans can be disbursed'], 400);
    }

    // Begin transaction
    $db->beginTransaction();

    // Update loan status to disbursed
    $stmt = $db->prepare("
        UPDATE loans 
        SET status = 'disbursed', disbursed_at = NOW()
        WHERE id = ?
    ");
    $stmt->execute([$loanId]);

    // Generate transaction reference
    $txnReference = generateTransactionReference();

    // Record disbursement transaction
    $stmt = $db->prepare("
        INSERT INTO transactions (
            transaction_reference, member_id, virtual_account_id, 
            type, amount, balance_after, description, loan_id
        ) VALUES (?, ?, ?, 'loan_disbursement', ?, 0, ?, ?)
    ");
    $stmt->execute([
        $txnReference,
        $loan['member_db_id'],
        $loan['va_id'],
        $loan['amount'],
        "Loan disbursement to " . $loan['receiving_account_number'] . " (" . $loan['receiving_bank_name'] . ")",
        $loanId
    ]);

    // Commit transaction
    $db->commit();

    jsonResponse([
        'success' => true,
        'message' => 'Loan disbursed successfully',
        'data' => [
            'loan_reference' => $loan['loan_reference'],
            'member' => $loan['full_name'],
            'amount' => floatval($loan['amount']),
            'disbursed_to' => [
                'account_number' => $loan['receiving_account_number'],
                'bank_name' => $loan['receiving_bank_name'],
                'account_name' => $loan['receiving_account_name']
            ],
            'transaction_reference' => $txnReference,
            'total_repayment' => floatval($loan['total_amount']),
            'monthly_repayment' => floatval($loan['monthly_repayment']),
            'repayment_plan' => $loan['repayment_plan'] . ' months'
        ]
    ]);

} catch (PDOException $e) {
    if ($db->inTransaction()) {
        $db->rollBack();
    }
    jsonResponse(['success' => false, 'message' => 'Disbursement failed: ' . $e->getMessage()], 500);
}
?>
