<?php
/**
 * Admin - Approve/Reject Loan API
 * PUT /api/admin/loans/approve.php
 */

require_once '../../../config/database.php';
require_once '../../../includes/cors.php';
require_once '../../../includes/helpers.php';
require_once '../../../includes/jwt.php';

if ($_SERVER['REQUEST_METHOD'] !== 'PUT') {
    jsonResponse(['success' => false, 'message' => 'Method not allowed'], 405);
}

// Authenticate admin
$jwt = new JWTHandler();
$userData = $jwt->authenticateAdmin();

$data = json_decode(file_get_contents("php://input"), true);

if (empty($data['loan_id'])) {
    jsonResponse(['success' => false, 'message' => 'Loan ID is required'], 400);
}

if (empty($data['action']) || !in_array($data['action'], ['approve', 'reject'])) {
    jsonResponse(['success' => false, 'message' => 'Valid action (approve, reject) is required'], 400);
}

$loanId = intval($data['loan_id']);
$action = $data['action'];

try {
    $database = new Database();
    $db = $database->getConnection();

    // Check if loan exists and is pending
    $stmt = $db->prepare("
        SELECT l.*, m.full_name, m.member_id as member_code, va.savings_balance
        FROM loans l
        JOIN members m ON l.member_id = m.id
        JOIN virtual_accounts va ON m.id = va.member_id
        WHERE l.id = ?
    ");
    $stmt->execute([$loanId]);
    $loan = $stmt->fetch();

    if (!$loan) {
        jsonResponse(['success' => false, 'message' => 'Loan not found'], 404);
    }

    if ($loan['status'] !== 'pending') {
        jsonResponse(['success' => false, 'message' => 'Only pending loans can be approved or rejected'], 400);
    }

    // Additional validation for approval
    if ($action === 'approve') {
        // Check if loan amount is within limits (200% of savings)
        $maxLoanAmount = calculateMaxLoanAmount($loan['savings_balance']);
        if ($loan['amount'] > $maxLoanAmount) {
            jsonResponse([
                'success' => false, 
                'message' => "Loan amount exceeds maximum allowed (₦" . number_format($maxLoanAmount, 2) . ")"
            ], 400);
        }
    }

    // Determine new status
    $newStatus = $action === 'approve' ? 'approved' : 'rejected';

    // Update loan status
    $stmt = $db->prepare("
        UPDATE loans 
        SET status = ?, approved_by = ?, approved_at = NOW()
        WHERE id = ?
    ");
    $stmt->execute([$newStatus, $userData['user_id'], $loanId]);

    jsonResponse([
        'success' => true,
        'message' => "Loan has been {$action}d successfully",
        'data' => [
            'loan_reference' => $loan['loan_reference'],
            'member' => $loan['full_name'],
            'amount' => floatval($loan['amount']),
            'new_status' => $newStatus
        ]
    ]);

} catch (PDOException $e) {
    jsonResponse(['success' => false, 'message' => 'Action failed: ' . $e->getMessage()], 500);
}
?>
