<?php
/**
 * Admin - Record Manual Deposit API
 * POST /api/admin/deposits/record.php
 */

require_once '../../../config/database.php';
require_once '../../../includes/cors.php';
require_once '../../../includes/helpers.php';
require_once '../../../includes/jwt.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonResponse(['success' => false, 'message' => 'Method not allowed'], 405);
}

// Authenticate admin
$jwt = new JWTHandler();
$userData = $jwt->authenticateAdmin();

$data = json_decode(file_get_contents("php://input"), true);

// Validate required fields
if (empty($data['member_id']) || empty($data['amount'])) {
    jsonResponse(['success' => false, 'message' => 'Member ID and amount are required'], 400);
}

$memberId = intval($data['member_id']);
$amount = floatval($data['amount']);
$description = isset($data['description']) ? sanitizeInput($data['description']) : 'Manual deposit by admin';

if ($amount <= 0) {
    jsonResponse(['success' => false, 'message' => 'Invalid amount'], 400);
}

try {
    $database = new Database();
    $db = $database->getConnection();

    // Get member and virtual account
    $stmt = $db->prepare("
        SELECT m.*, va.id as va_id, va.account_number, va.savings_balance
        FROM members m
        JOIN virtual_accounts va ON m.id = va.member_id
        WHERE m.id = ?
    ");
    $stmt->execute([$memberId]);
    $member = $stmt->fetch();

    if (!$member) {
        jsonResponse(['success' => false, 'message' => 'Member not found'], 404);
    }

    // Begin transaction
    $db->beginTransaction();

    $remainingAmount = $amount;
    $loanRepaid = 0;
    $addedToSavings = 0;
    $transactions = [];

    // Check for active loan
    $stmt = $db->prepare("
        SELECT * FROM loans 
        WHERE member_id = ? AND status IN ('disbursed', 'repaying')
        ORDER BY disbursed_at ASC LIMIT 1
    ");
    $stmt->execute([$memberId]);
    $activeLoan = $stmt->fetch();

    if ($activeLoan && $activeLoan['amount_remaining'] > 0) {
        // Calculate repayment amount
        $repaymentAmount = min($remainingAmount, $activeLoan['amount_remaining']);
        $remainingAmount -= $repaymentAmount;
        $loanRepaid = $repaymentAmount;

        // Generate transaction reference
        $txnReference = generateTransactionReference();

        // Update loan
        $newAmountPaid = $activeLoan['amount_paid'] + $repaymentAmount;
        $newAmountRemaining = $activeLoan['amount_remaining'] - $repaymentAmount;
        $newLoanStatus = $newAmountRemaining <= 0 ? 'completed' : 'repaying';
        $completedAt = $newLoanStatus === 'completed' ? date('Y-m-d H:i:s') : null;

        $stmt = $db->prepare("
            UPDATE loans 
            SET amount_paid = ?, amount_remaining = ?, status = ?, completed_at = ?
            WHERE id = ?
        ");
        $stmt->execute([$newAmountPaid, $newAmountRemaining, $newLoanStatus, $completedAt, $activeLoan['id']]);

        // Record loan repayment transaction
        $stmt = $db->prepare("
            INSERT INTO transactions (
                transaction_reference, member_id, virtual_account_id, 
                type, amount, balance_after, description, loan_id
            ) VALUES (?, ?, ?, 'loan_repayment', ?, ?, ?, ?)
        ");
        $stmt->execute([
            $txnReference,
            $memberId,
            $member['va_id'],
            $repaymentAmount,
            $member['savings_balance'],
            "Loan repayment for " . $activeLoan['loan_reference'] . " (recorded by admin)",
            $activeLoan['id']
        ]);
        $transactionId = $db->lastInsertId();

        // Record in loan_repayments
        $stmt = $db->prepare("
            INSERT INTO loan_repayments (
                loan_id, transaction_id, amount, principal_paid, interest_paid, balance_after
            ) VALUES (?, ?, ?, ?, ?, ?)
        ");
        
        $totalPrincipal = $activeLoan['amount'];
        $totalInterest = $activeLoan['total_amount'] - $activeLoan['amount'];
        $ratio = $repaymentAmount / $activeLoan['total_amount'];
        $principalPaid = $totalPrincipal * $ratio;
        $interestPaid = $totalInterest * $ratio;
        
        $stmt->execute([
            $activeLoan['id'],
            $transactionId,
            $repaymentAmount,
            $principalPaid,
            $interestPaid,
            $newAmountRemaining
        ]);

        $transactions[] = [
            'type' => 'loan_repayment',
            'reference' => $txnReference,
            'amount' => $repaymentAmount,
            'loan_reference' => $activeLoan['loan_reference'],
            'loan_balance' => $newAmountRemaining
        ];
    }

    // Add remaining amount to savings
    if ($remainingAmount > 0) {
        $addedToSavings = $remainingAmount;
        $newSavingsBalance = $member['savings_balance'] + $remainingAmount;

        // Update savings balance
        $stmt = $db->prepare("
            UPDATE virtual_accounts SET savings_balance = ? WHERE id = ?
        ");
        $stmt->execute([$newSavingsBalance, $member['va_id']]);

        // Generate transaction reference
        $txnReference = generateTransactionReference();

        // Record deposit transaction
        $stmt = $db->prepare("
            INSERT INTO transactions (
                transaction_reference, member_id, virtual_account_id, 
                type, amount, balance_after, description
            ) VALUES (?, ?, ?, 'deposit', ?, ?, ?)
        ");
        $stmt->execute([
            $txnReference,
            $memberId,
            $member['va_id'],
            $remainingAmount,
            $newSavingsBalance,
            $description
        ]);

        $transactions[] = [
            'type' => 'deposit',
            'reference' => $txnReference,
            'amount' => $remainingAmount,
            'new_balance' => $newSavingsBalance
        ];
    }

    // Commit transaction
    $db->commit();

    jsonResponse([
        'success' => true,
        'message' => 'Deposit recorded successfully',
        'data' => [
            'member' => [
                'member_id' => $member['member_id'],
                'full_name' => $member['full_name']
            ],
            'total_deposited' => $amount,
            'loan_repaid' => $loanRepaid,
            'added_to_savings' => $addedToSavings,
            'transactions' => $transactions
        ]
    ]);

} catch (PDOException $e) {
    if ($db->inTransaction()) {
        $db->rollBack();
    }
    jsonResponse(['success' => false, 'message' => 'Failed to record deposit: ' . $e->getMessage()], 500);
}
?>
