<?php
/**
 * Admin Dashboard Statistics API
 * GET /api/admin/dashboard.php
 */

require_once '../../config/database.php';
require_once '../../includes/cors.php';
require_once '../../includes/helpers.php';
require_once '../../includes/jwt.php';

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    jsonResponse(['success' => false, 'message' => 'Method not allowed'], 405);
}

// Authenticate admin
$jwt = new JWTHandler();
$userData = $jwt->authenticateAdmin();

try {
    $database = new Database();
    $db = $database->getConnection();

    // Total members by status
    $stmt = $db->query("
        SELECT status, COUNT(*) as count 
        FROM members 
        GROUP BY status
    ");
    $memberStats = $stmt->fetchAll();
    $membersByStatus = [];
    $totalMembers = 0;
    foreach ($memberStats as $stat) {
        $membersByStatus[$stat['status']] = intval($stat['count']);
        $totalMembers += intval($stat['count']);
    }

    // Total savings
    $stmt = $db->query("SELECT SUM(savings_balance) as total FROM virtual_accounts");
    $totalSavings = floatval($stmt->fetch()['total'] ?? 0);

    // Loan statistics
    $stmt = $db->query("
        SELECT status, COUNT(*) as count, SUM(amount) as total_amount, SUM(amount_remaining) as total_remaining
        FROM loans 
        GROUP BY status
    ");
    $loanStats = $stmt->fetchAll();
    $loansByStatus = [];
    $totalLoansAmount = 0;
    $totalOutstanding = 0;
    foreach ($loanStats as $stat) {
        $loansByStatus[$stat['status']] = [
            'count' => intval($stat['count']),
            'amount' => floatval($stat['total_amount'] ?? 0)
        ];
        $totalLoansAmount += floatval($stat['total_amount'] ?? 0);
        if (in_array($stat['status'], ['disbursed', 'repaying'])) {
            $totalOutstanding += floatval($stat['total_remaining'] ?? 0);
        }
    }

    // Recent transactions (last 7 days)
    $stmt = $db->query("
        SELECT type, COUNT(*) as count, SUM(amount) as total
        FROM transactions
        WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
        GROUP BY type
    ");
    $recentTransactions = $stmt->fetchAll();

    // Pending approvals count
    $stmt = $db->query("SELECT COUNT(*) as count FROM members WHERE status = 'pending'");
    $pendingMembers = intval($stmt->fetch()['count']);

    $stmt = $db->query("SELECT COUNT(*) as count FROM loans WHERE status = 'pending'");
    $pendingLoans = intval($stmt->fetch()['count']);

    $stmt = $db->query("SELECT COUNT(*) as count FROM loans WHERE status = 'approved'");
    $approvedLoansAwaitingDisbursement = intval($stmt->fetch()['count']);

    // Top 5 recent registrations
    $stmt = $db->query("
        SELECT member_id, full_name, email, status, created_at 
        FROM members 
        ORDER BY created_at DESC 
        LIMIT 5
    ");
    $recentRegistrations = $stmt->fetchAll();

    // Top 5 recent loan applications
    $stmt = $db->query("
        SELECT l.loan_reference, l.amount, l.status, l.applied_at, m.full_name
        FROM loans l
        JOIN members m ON l.member_id = m.id
        ORDER BY l.applied_at DESC 
        LIMIT 5
    ");
    $recentLoans = $stmt->fetchAll();

    jsonResponse([
        'success' => true,
        'data' => [
            'members' => [
                'total' => $totalMembers,
                'by_status' => $membersByStatus,
                'pending_approval' => $pendingMembers
            ],
            'savings' => [
                'total' => $totalSavings
            ],
            'loans' => [
                'by_status' => $loansByStatus,
                'total_disbursed' => $totalLoansAmount,
                'total_outstanding' => $totalOutstanding,
                'pending_approval' => $pendingLoans,
                'awaiting_disbursement' => $approvedLoansAwaitingDisbursement
            ],
            'recent_transactions' => $recentTransactions,
            'recent_registrations' => $recentRegistrations,
            'recent_loans' => $recentLoans
        ]
    ]);

} catch (PDOException $e) {
    jsonResponse(['success' => false, 'message' => 'Failed to get dashboard: ' . $e->getMessage()], 500);
}
?>
